// Copyright 2014 Google Inc. All Rights Reserved.

#include "MediaPlaybackStatusEndpoint.h"

void MediaPlaybackStatusEndpoint::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    service->mutable_media_playback_service();
}

int MediaPlaybackStatusEndpoint::routeMessage(
        uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case MEDIA_PLAYBACK_STATUS: {
            MediaPlaybackStatus status;
            if (PARSE_PROTO(status, ptr, len)) {
                handleMediaPlaybackStatus(status);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case MEDIA_PLAYBACK_METADATA: {
            MediaPlaybackMetadata metadata;
            if (PARSE_PROTO(metadata, ptr, len)) {
                handleMediaPlaybackMetadata(metadata);
                ret = STATUS_SUCCESS;
            }
            break;
        }
    }
    return ret;
}

int MediaPlaybackStatusEndpoint::reportAction(uint32_t action) {
    if (InstrumentClusterInput::InstrumentClusterAction_IsValid(action)) {
      InstrumentClusterInput input;
      input.set_action((InstrumentClusterInput_InstrumentClusterAction)action);
      IoBuffer buf;
      mRouter->marshallProto(MEDIA_PLAYBACK_INPUT, input, &buf);
      queueOutgoing(buf.raw(), buf.size());
      return STATUS_SUCCESS;
    }
    return STATUS_INVALID_INPUT;
}

void MediaPlaybackStatusEndpoint::handleMediaPlaybackStatus(const MediaPlaybackStatus &status) {
    if (mCallbacks != NULL) {
        struct MediaPlaybackStatusStruct status_struct;
        status_struct.state = status.state();
        status_struct.media_source = status.media_source();
        status_struct.playback_seconds = status.playback_seconds();
        status_struct.shuffle = status.shuffle();
        status_struct.repeat = status.repeat();
        status_struct.repeat_one = status.repeat_one();
        mCallbacks->mediaPlaybackStatusCallback(status_struct);
    }
}

void MediaPlaybackStatusEndpoint::handleMediaPlaybackMetadata(
    const MediaPlaybackMetadata &metadata) {
    if (mCallbacks != NULL) {
        struct MediaPlaybackMetadataStruct metadata_struct;
        metadata_struct.song = metadata.song();
        metadata_struct.album = metadata.album();
        metadata_struct.artist = metadata.artist();
        metadata_struct.album_art = metadata.album_art();
        metadata_struct.playlist = metadata.playlist();
        metadata_struct.duration_seconds = metadata.duration_seconds();
        metadata_struct.rating = metadata.rating();
        mCallbacks->mediaPlaybackMetadataCallback(metadata_struct);
    }
}
